unit XPXMLDOM;

{
  Adapter for TurboPower XMLPartner v2.5 for Delphi's XML DOM interfaces.

  Written by Keith Wood (kbwood@iprimus.com.au)
  Version 1.0 - 21 August 2002.
}

interface

uses
  Windows, SysUtils, Classes, XMLDOM, XpBase, XpDOM, XpvXSLPr, XpvFlXML;

const
  SXMLPartner = 'XML Partner';  { Do not localize }

type

{ IXpNodeRef }

  IXpNodeRef = interface
    ['{84DE09B6-A51A-4244-8F26-A789C04B1BD0}']
    function GetXMLDOMNode: TXpNode;
  end;

{ TXMLPDOMInterface }

  TXMLPDOMInterface = class(TInterfacedObject)
  end;

{ TXMLPDOMImplementation }

  TXMLPDOMImplementation = class(TXMLPDOMInterface, IDOMImplementation)
  private
    FDOMImpl: TXpDOMImplementation;
  protected
    { IDOMImplementation }
    function hasFeature(const feature, version: DOMString): WordBool;
    function createDocumentType(const qualifiedName, publicId,
      systemId: DOMString): IDOMDocumentType; safecall;
    function createDocument(const namespaceURI, qualifiedName: DOMString;
      doctype: IDOMDocumentType): IDOMDocument; safecall;
  public
    constructor Create(DOMImplementation: TXpDomImplementation);
    destructor Destroy; override;
    property DOMImpl: TXpDOMImplementation read FDOMImpl;
  end;

{ TXMLPDOMNode }

  TXMLPDOMNodeClass = class of TXMLPDOMNode;

  TXMLPDOMDocument = class;

  TXMLPDOMNode = class(TXMLPDOMInterface,
    IXpNodeRef, IDOMNode, IDOMNodeEx, IDOMNodeSelect)
  private
    FAttributes: IDOMNamedNodeMap;
    FChildNodes: IDOMNodeList;
    FNode: TXpNode;
    FOwnerDocument: TXMLPDOMDocument;
  protected
    { IXpNodeRef }
    function GetXMLDOMNode: TXpNode;
    { IDOMNode }
    function get_nodeName: DOMString; virtual; safecall;
    function get_nodeValue: DOMString; safecall;
    procedure set_nodeValue(value: DOMString);
    function get_nodeType: DOMNodeType; virtual; safecall;
    function get_parentNode: IDOMNode; safecall;
    function get_childNodes: IDOMNodeList; virtual; safecall;
    function get_firstChild: IDOMNode; safecall;
    function get_lastChild: IDOMNode; safecall;
    function get_previousSibling: IDOMNode; safecall;
    function get_nextSibling: IDOMNode; safecall;
    function get_attributes: IDOMNamedNodeMap; safecall;
    function get_ownerDocument: IDOMDocument; safecall;
    function get_namespaceURI: DOMString; safecall;
    function get_prefix: DOMString; safecall;
    function get_localName: DOMString; safecall;
    function insertBefore(const newChild, refChild: IDOMNode): IDOMNode;
      safecall;
    function replaceChild(const newChild, oldChild: IDOMNode): IDOMNode;
      safecall;
    function removeChild(const childNode: IDOMNode): IDOMNode; safecall;
    function appendChild(const newChild: IDOMNode): IDOMNode; safecall;
    function hasChildNodes: WordBool; virtual; safecall;
    function cloneNode(deep: WordBool): IDOMNode; safecall;
    procedure normalize;
    function supports(const feature, version: DOMString): WordBool;
    { IDOMNodeEx }
    function get_text: DOMString; safecall;
    function get_xml: DOMString; virtual; safecall;
    procedure set_text(const Value: DOMString); safecall;
    procedure transformNode(const stylesheet: IDOMNode; var output: WideString);
      overload;
    procedure transformNode(const stylesheet: IDOMNode;
      const output: IDOMDocument); overload;
    { IDOMNodeSelect }
    function selectNode(const nodePath: WideString): IDOMNode; safecall;
    function selectNodes(const nodePath: WideString): IDOMNodeList; safecall;
  public
    constructor Create(const Node: TXpNode; const Document: TXMLPDOMDocument);
      virtual;
    property Node: TXpNode read FNode;
  end;

{ TXMLPDOMNodeList }

  TXMLPDOMNodeList = class(TXMLPDOMInterface, IDOMNodeList)
  private
    FNodeList: TXpNodeList;
    FOwnerDocument: TXMLPDOMDocument;
  protected
    { IDOMNodeList }
    function get_item(index: Integer): IDOMNode; safecall;
    function get_length: Integer; safecall;
  public
    constructor Create(const NodeList: TXpNodeList;
      const Document: TXMLPDOMDocument);
    property NodeList: TXpNodeList read FNodeList;
  end;

{ TXMLPDOMNamedNodeMap }

  TXMLPDOMNamedNodeMap = class(TXMLPDOMInterface, IDOMNamedNodeMap)
  private
    FNamedNodeMap: TXpNamedNodeMap;
    FOwnerDocument: TXMLPDOMDocument;
  protected
    { IDOMNamedNodeMap }
    function get_item(index: Integer): IDOMNode; safecall;
    function get_length: Integer;
    function getNamedItem(const name: DOMString): IDOMNode; safecall;
    function setNamedItem(const newItem: IDOMNode): IDOMNode; safecall;
    function removeNamedItem(const name: DOMString): IDOMNode; safecall;
    function getNamedItemNS(const namespaceURI, localName: DOMString):
      IDOMNode; safecall;
    function setNamedItemNS(const arg: IDOMNode): IDOMNode; safecall;
    function removeNamedItemNS(const namespaceURI, localName: DOMString):
      IDOMNode; safecall;
  public
    constructor Create(const NamedNodeMap: TXpNamedNodeMap;
      const Document: TXMLPDOMDocument);
    property NamedNodeMap: TXpNamedNodeMap read FNamedNodeMap;
  end;

{ TXMLPDOMCharacterData }

  TXMLPDOMCharacterData = class(TXMLPDOMNode, IDOMCharacterData)
  private
    function GetCharacterData: TXpCharacterData;
  protected
    { IDOMCharacterData }
    function get_data: DOMString;
    procedure set_data(const data: DOMString);
    function get_length: Integer;
    function substringData(offset, count: Integer): DOMString;
    procedure appendData(const data: DOMString);
    procedure insertData(offset: Integer; const data: DOMString);
    procedure deleteData(offset, count: Integer);
    procedure replaceData(offset, count: Integer; const data: DOMString);
  public
    property CharacterData: TXpCharacterData read GetCharacterData;
  end;

{ TXMLPDOMAttr }

  TXMLPDOMAttr = class(TXMLPDOMNode, IDOMAttr)
  private
    function GetAttribute: TXpAttribute;
  protected
    { IDOMAttr }
    function get_name: DOMString;
    function get_specified: WordBool;
    function get_value: DOMString;
    procedure set_value(const attributeValue: DOMString);
    function get_ownerElement: IDOMElement;
  public
    property Attribute: TXpAttribute read GetAttribute;
  end;

{ TXMLPDOMElement }

  TXMLPDOMElement = class(TXMLPDOMNode, IDOMElement)
  private
    function GetElement: TXpElement;
  protected
    { IDOMElement }
    function get_tagName: DOMString; safecall;
    function getAttribute(const name: DOMString): DOMString; safecall;
    procedure setAttribute(const name, value: DOMString);
    procedure removeAttribute(const name: DOMString);
    function getAttributeNode(const name: DOMString): IDOMAttr; safecall;
    function setAttributeNode(const newAttr: IDOMAttr): IDOMAttr; safecall;
    function removeAttributeNode(const oldAttr: IDOMAttr): IDOMAttr; safecall;
    function getElementsByTagName(const name: DOMString): IDOMNodeList; safecall;
    function getAttributeNS(const namespaceURI, localName: DOMString):
      DOMString; safecall;
    procedure setAttributeNS(const namespaceURI, qualifiedName, value: DOMString);
    procedure removeAttributeNS(const namespaceURI, localName: DOMString);
    function getAttributeNodeNS(const namespaceURI, localName: DOMString):
      IDOMAttr; safecall;
    function setAttributeNodeNS(const newAttr: IDOMAttr): IDOMAttr; safecall;
    function getElementsByTagNameNS(const namespaceURI,
      localName: DOMString): IDOMNodeList; safecall;
    function hasAttribute(const name: DOMString): WordBool; safecall;
    function hasAttributeNS(const namespaceURI, localName: DOMString): WordBool;
    procedure normalize;
  public
    property Element: TXpElement read GetElement;
  end;

{ TXMLPDOMText }

  TXMLPDOMText = class(TXMLPDOMCharacterData, IDOMText)
  protected
    function splitText(offset: Integer): IDOMText; safecall;
  end;

{ TXMLPDOMComment }

  TXMLPDOMComment = class(TXMLPDOMCharacterData, IDOMComment)
  end;

{ TXMLPDOMCDATASection }

  TXMLPDOMCDATASection = class(TXMLPDOMText, IDOMCDATASection)
  end;

{ TXMLPDOMDocumentType }

  TXMLPDOMDocumentType = class(TXMLPDOMNode, IDOMDocumentType)
  private
    FChildren: IDOMNodeList;
    FEntities: IDOMNamedNodeMap;
    FNotations: IDOMNamedNodeMap;
    function GetDocumentType: TXpDocumentType;
  protected
    function get_childNodes: IDOMNodeList; override; safecall;
    function hasChildNodes: WordBool; override; safecall;
    { IDOMDocumentType }
    function get_name: DOMString; safecall;
    function get_entities: IDOMNamedNodeMap; safecall;
    function get_notations: IDOMNamedNodeMap; safecall;
    function get_publicId: DOMString; safecall;
    function get_systemId: DOMString; safecall;
    function get_internalSubset: DOMString; safecall;
  public
    constructor Create(const Node: TXpNode; const Document: TXMLPDOMDocument);
      override;
    property DocumentType: TXpDocumentType read GetDocumentType;
  end;

{ TXMLPDOMNotation }

  TXMLPDOMNotation = class(TXMLPDOMNode, IDOMNotation)
  private
    function GetNotation: TXpDTDNotation;
  protected
    function get_nodeType: DOMNodeType; override; safecall;
    { IDOMNotation }
    function get_publicId: DOMString; safecall;
    function get_systemId: DOMString; safecall;
  public
    property Notation: TXpDTDNotation read GetNotation;
  end;

{ TXMLPDOMEntity }

  TXMLPDOMEntity = class(TXMLPDOMNode, IDOMEntity)
  private
    function GetEntity: TXpDTDEntity;
  protected
    function get_nodeType: DOMNodeType; override; safecall;
    { IDOMEntity }
    function get_publicId: DOMString; safecall;
    function get_systemId: DOMString; safecall;
    function get_notationName: DOMString; safecall;
  public
    property Entity: TXpDTDEntity read GetEntity;
  end;

{ TXMLPDOMEntityReference }

  TXMLPDOMEntityReference = class(TXMLPDOMNode, IDOMEntityReference)
  end;

{ TXMLPDOMProcessingInstruction }

  TXMLPDOMProcessingInstruction = class(TXMLPDOMNode, IDOMProcessingInstruction)
  private
    function GetProcessingInstruction: TXpProcessingInstruction;
  protected
    { IDOMProcessingInstruction }
    function get_target: DOMString; safecall;
    function get_data: DOMString; safecall;
    procedure set_data(const value: DOMString);
  public
    property ProcessingInstruction: TXpProcessingInstruction
      read GetProcessingInstruction;
  end;

{ TXMLPDOMDocumentFragment }

  TXMLPDOMDocumentFragment = class(TXMLPDOMNode, IDOMDocumentFragment)
  end;

{ TXMLPDOMDocument }

  TXMLPDOMDocument = class(TXMLPDOMNode, IDOMDocument, IDOMParseOptions,
    IDOMPersist, IDOMParseError, IDOMXMLProlog)
  private
    FEncoding: DOMString;
    FGotProlog: Boolean;
    FStandalone: DOMString;
    FValidate: Boolean;
    FVersion: DOMString;
    FXMLObjModel: TXpObjModel;
    function GetDocument: TXpDocument;
  protected
    procedure GetProlog;
    { IDOMNode }
    function get_nodeName: DOMString; override; safecall;
    { IDOMDocument }
    function get_doctype: IDOMDocumentType; safecall;
    function get_domImplementation: IDOMImplementation; safecall;
    function get_documentElement: IDOMElement; safecall;
    procedure set_documentElement(const IDOMElement: IDOMElement);
    function createElement(const tagName: DOMString): IDOMElement; safecall;
    function createDocumentFragment: IDOMDocumentFragment; safecall;
    function createTextNode(const data: DOMString): IDOMText; safecall;
    function createComment(const data: DOMString): IDOMComment; safecall;
    function createCDATASection(const data: DOMString): IDOMCDATASection; safecall;
    function createProcessingInstruction(const target,
      data: DOMString): IDOMProcessingInstruction; safecall;
    function createAttribute(const name: DOMString): IDOMAttr; safecall;
    function createEntityReference(const name: DOMString): IDOMEntityReference; safecall;
    function getElementsByTagName(const tagName: DOMString): IDOMNodeList; safecall;
    function importNode(importedNode: IDOMNode; deep: WordBool): IDOMNode; safecall;
    function createElementNS(const namespaceURI,
      qualifiedName: DOMString): IDOMElement; safecall;
    function createAttributeNS(const namespaceURI,
      qualifiedName: DOMString): IDOMAttr; safecall;
    function getElementsByTagNameNS(const namespaceURI,
      localName: DOMString): IDOMNodeList; safecall;
    function getElementById(const elementId: DOMString): IDOMElement;
    { IDOMParseOptions }
    function get_async: Boolean;
    function get_preserveWhiteSpace: Boolean;
    function get_resolveExternals: Boolean;
    function get_validate: Boolean;
    procedure set_async(Value: Boolean);
    procedure set_preserveWhiteSpace(Value: Boolean);
    procedure set_resolveExternals(Value: Boolean);
    procedure set_validate(Value: Boolean);
    { IDOMPersist }
    function get_xml: DOMString; override; safecall;
    function asyncLoadState: Integer; safecall;
    function load(source: OleVariant): WordBool; safecall;
    function loadFromStream(const stream: TStream): WordBool; safecall;
    function loadxml(const Value: DOMString): WordBool; safecall;
    procedure save(destination: OleVariant); safecall;
    procedure saveToStream(const stream: TStream); safecall;
    procedure set_OnAsyncLoad(const Sender: TObject;
      EventHandler: TAsyncEventHandler); safecall;
    { IDOMParseError }
    function get_errorCode: Integer;
    function get_url: WideString; safecall;
    function get_reason: WideString; safecall;
    function get_srcText: WideString; safecall;
    function get_line: Integer;
    function get_linepos: Integer;
    function get_filepos: Integer;
    { IDOMXMLProlog }
    function get_Encoding: DOMString; safecall;
    function get_Standalone: DOMString; safecall;
    function get_Version: DOMString; safecall;
    procedure set_Encoding(const Value: DOMString); safecall;
    procedure set_Standalone(const Value: DOMString); safecall;
    procedure set_Version(const Value: DOMString); safecall;
  public
    constructor Create(const Node: TXpNode; const Document: TXMLPDOMDocument);
      override;
    destructor Destroy; override;
    property Document: TXpDocument read GetDocument;
  end;

{ TXMLPDOMImplementationFactory }

  TXMLPDOMImplementationFactory = class(TDOMVendor)
  public
    function DOMImplementation: IDOMImplementation; override;
    function Description: String; override;
  end;

var
  XPXML_DOM: TXMLPDOMImplementationFactory;

implementation

resourcestring
  SNodeExpected = 'Node cannot be null';

{ Utility Functions -----------------------------------------------------------}

function MakeNode(const Node: TXpNode; const Document: TXMLPDOMDocument):
  IDOMNode;
const
  NodeClasses: array [ELEMENT_NODE..NOTATION_NODE] of TXMLPDOMNodeClass =
    (TXMLPDOMElement, TXMLPDOMAttr, TXMLPDOMText, TXMLPDOMCDataSection,
     TXMLPDOMEntityReference, TXMLPDOMEntity, TXMLPDOMProcessingInstruction,
     TXMLPDOMComment, TXMLPDOMDocument, TXMLPDOMDocumentType,
     TXMLPDOMDocumentFragment, TXMLPDOMNotation);
begin
  if Assigned(Node) then
    case Node.nodeType of
      ENTITY_DECL_NODE:   Result := TXMLPDOMEntity.Create(Node, Document);
      NOTATION_DECL_NODE: Result := TXMLPDOMNotation.Create(Node, Document);
      else                Result :=
                            NodeClasses[Node.nodeType].Create(Node, Document);
    end
  else
    Result := nil;
end;

function MakeNodeList(const NodeList: TXpNodeList;
  const Document: TXMLPDOMDocument): IDOMNodeList;
begin
  Result := TXMLPDOMNodeList.Create(NodeList, Document);
end;

function MakeNamedNodeMap(const NamedNodeMap: TXpNamedNodeMap;
  const Document: TXMLPDOMDocument): IDOMNamedNodeMap;
begin
  Result := TXMLPDOMNamedNodeMap.Create(NamedNodeMap, Document);
end;

function GetNode(const Node: IDOMNode): TXpNode;
begin
  if not Assigned(Node) then
    raise DOMException.Create(SNodeExpected);
  Result := (Node as IXpNodeRef).GetXMLDOMNode;
end;

{ TXMLPDOMImplementation ------------------------------------------------------}

constructor TXMLPDOMImplementation.Create(
  DOMImplementation: TXpDomImplementation);
begin
  inherited Create;
  FDOMImpl := DOMImplementation;
end;

destructor TXMLPDOMImplementation.Destroy;
begin
  FDOMImpl.Free;
  inherited Destroy;
end;

function TXMLPDOMImplementation.createDocument(const namespaceURI,
  qualifiedName: DOMString; doctype: IDOMDocumentType): IDOMDocument;
var
  DocumentType: TXpDocumentType;
  Document: TXMLPDOMDocument;
begin
  if Assigned(doctype) then
    DocumentType := GetNode(docType) as TXpDocumentType
  else
    DocumentType := nil;
  Document := TXMLPDOMDocument.Create(FDOMImpl.CreateDocument(
    namespaceURI, qualifiedName, DocumentType), nil);
  Document.FOwnerDocument := Document;
  Result   := Document;
end;

function TXMLPDOMImplementation.createDocumentType(const qualifiedName,
  publicId, systemId: DOMString): IDOMDocumentType;
begin
  Result := TXMLPDOMDocumentType.Create(FDOMImpl.CreateDocumentType(
    qualifiedName, publicId, systemId), nil);
end;

function TXMLPDOMImplementation.hasFeature(
  const feature, version: DOMString): WordBool;
begin
  Result := DOMImpl.hasFeature(feature, version);
end;

{ TXMLPDOMNode ----------------------------------------------------------------}

constructor TXMLPDOMNode.Create(const Node: TXpNode;
  const Document: TXMLPDOMDocument);
begin
  Assert(Assigned(Node));
  FNode          := Node;
  FOwnerDocument := Document;
  inherited Create;
end;

function TXMLPDOMNode.appendChild(const newChild: IDOMNode): IDOMNode;
begin
  Node.appendChild(GetNode(newChild));
  Result := newChild
end;

function TXMLPDOMNode.cloneNode(Deep: WordBool): IDOMNode;
begin
  Result := MakeNode(Node.CloneNode(deep), FOwnerDocument);
end;

function TXMLPDOMNode.get_attributes: IDOMNamedNodeMap;
begin
  if not Assigned(FAttributes) and Assigned(Node.Attributes) then
    FAttributes := MakeNamedNodeMap(Node.Attributes, FOwnerDocument);
  Result := FAttributes;
end;

function TXMLPDOMNode.get_childNodes: IDOMNodeList;
begin
  if not Assigned(FChildNodes) then
    FChildNodes := MakeNodeList(Node.ChildNodes, FOwnerDocument);
  Result := FChildNodes;
end;

function TXMLPDOMNode.get_firstChild: IDOMNode;
begin
  Result := MakeNode(Node.FirstChild, FOwnerDocument);
end;

function TXMLPDOMNode.get_lastChild: IDOMNode;
begin
  Result := MakeNode(Node.LastChild, FOwnerDocument);
end;

function TXMLPDOMNode.get_localName: DOMString;
begin
  Result := Node.LocalName;
end;

function TXMLPDOMNode.get_namespaceURI: DOMString;
begin
  Result := Node.NamespaceURI;
end;

function TXMLPDOMNode.get_nextSibling: IDOMNode;
begin
  Result := MakeNode(Node.NextSibling, FOwnerDocument);
end;

function TXMLPDOMNode.get_nodeName: DOMString;
begin
  Result := Node.NodeName;
end;

function TXMLPDOMNode.get_nodeType: DOMNodeType;
begin
  Result := Node.NodeType;
end;

function TXMLPDOMNode.get_nodeValue: DOMString;
begin
  Result := Node.NodeValue;
end;

function TXMLPDOMNode.get_ownerDocument: IDOMDocument;
begin
  Result := FOwnerDocument;
end;

function TXMLPDOMNode.get_parentNode: IDOMNode;
begin
  Result := MakeNode(Node.ParentNode, FOwnerDocument);
end;

function TXMLPDOMNode.get_prefix: DOMString;
begin
  Result := Node.Prefix;
end;

function TXMLPDOMNode.get_previousSibling: IDOMNode;
begin
  Result := MakeNode(Node.PreviousSibling, FOwnerDocument);
end;

function TXMLPDOMNode.hasChildNodes: WordBool;
begin
  Result := Node.HasChildNodes;
end;

function TXMLPDOMNode.insertBefore(const newChild, refChild: IDOMNode): IDOMNode;
begin
  Node.InsertBefore(GetNode(newChild), GetNode(refChild));
  Result := newChild;
end;

procedure TXMLPDOMNode.normalize;
begin
  Node.Normalize;
end;

function TXMLPDOMNode.removeChild(const childNode: IDOMNode): IDOMNode;
begin
  Result := MakeNode(Node.RemoveChild(GetNode(childNode)), FOwnerDocument);
end;

function TXMLPDOMNode.replaceChild(const newChild, oldChild: IDOMNode): IDOMNode;
begin
  Result := MakeNode(Node.ReplaceChild(GetNode(newChild), GetNode(oldChild)),
    FOwnerDocument);
end;

procedure TXMLPDOMNode.set_nodeValue(value: DOMString);
begin
  Node.NodeValue := value;
end;

function TXMLPDOMNode.supports(const feature, version: DOMString): WordBool;
begin
  Result := Node.IsSupported(feature, version);
end;

function TXMLPDOMNode.GetXMLDOMNode: TXpNode;
begin
  Result := Node;
end;

function TXMLPDOMNode.selectNode(const nodePath: WideString): IDOMNode;
begin
  Result := MakeNode(Node.SelectSingleNode(nodePath), FOwnerDocument);
end;

function TXMLPDOMNode.selectNodes(const nodePath: WideString): IDOMNodeList;
begin
  Result := MakeNodeList(Node.SelectNodes(nodePath), FOwnerDocument);
end;

{ IDOMNodeEx Interface }

function TXMLPDOMNode.get_text: DOMString;
begin
  Result := Node.Text;
end;

procedure TXMLPDOMNode.set_text(const Value: DOMString);
var
  Index: Integer;
begin
  for Index := Node.ChildNodes.Length - 1 downto 0 do
    Node.RemoveChild(Node.ChildNodes.Item(0));
  Node.AppendChild(Node.OwnerDocument.CreateTextNode(Value));
end;

function TXMLPDOMNode.get_xml: DOMString;
begin
  Result := Node.XmlDocument;
end;

procedure TXMLPDOMNode.transformNode(const stylesheet: IDOMNode;
  var output: WideString);
var
  XSLProc: TXpXSLProcessor;
begin
  XSLProc := TXpXSLProcessor.Create(nil);
  with XSLProc do
    try
      Filter      := TXpFilterXML.Create(XSLProc);
      XmlObjModel := FOwnerDocument.FXMLObjModel;
      StyleData   := GetNode(stylesheet).XmlDocument;
      if ApplyStyle then
        output := TXpFilterXML(Filter).XMLDocument
      else
        raise DOMException.Create(Errors.Text);
    finally
      Free;
    end;
end;

procedure TXMLPDOMNode.transformNode(const stylesheet: IDOMNode;
  const output: IDOMDocument);
var
  XSLOutput: WideString;
  XMLObjModel: TXpObjModel;
begin
  transformNode(stylesheet, XSLOutput);
  XMLObjModel := TXpObjModel.Create(nil);
  with XMLObjModel do
    try
      LoadMemory(XSLOutput, Length(XSLOutput));
      output.importNode(MakeNode(Document, FOwnerDocument), True);
    finally
      Free;
    end;
end;

{ TXMLPDOMNodeList ------------------------------------------------------------}

constructor TXMLPDOMNodeList.Create(const NodeList: TXpNodeList;
      const Document: TXMLPDOMDocument);
begin
  inherited Create;
  FNodeList      := NodeList;
  FOwnerDocument := Document;
end;

function TXMLPDOMNodeList.get_item(index: Integer): IDOMNode;
begin
  Result := MakeNode(NodeList.Item(index), FOwnerDocument);
end;

function TXMLPDOMNodeList.get_length: Integer;
begin
  Result := NodeList.Length;
end;

{ TXMLPDOMNamedNodeMap --------------------------------------------------------}

constructor TXMLPDOMNamedNodeMap.Create(const NamedNodeMap: TXpNamedNodeMap;
      const Document: TXMLPDOMDocument);
begin
  inherited Create;
  FNamedNodeMap  := NamedNodeMap;
  FOwnerDocument := Document;
end;

function TXMLPDOMNamedNodeMap.get_item(index: Integer): IDOMNode;
begin
  Result := MakeNode(NamedNodeMap.Item(index), FOwnerDocument);
end;

function TXMLPDOMNamedNodeMap.get_length: Integer;
begin
  Result := NamedNodeMap.Length;
end;

function TXMLPDOMNamedNodeMap.getNamedItem(const name: DOMString): IDOMNode;
begin
  Result := MakeNode(NamedNodeMap.GetNamedItem(name), FOwnerDocument);
end;

function TXMLPDOMNamedNodeMap.getNamedItemNS(
  const namespaceURI, localName: DOMString): IDOMNode;
begin
  Result := MakeNode(NamedNodeMap.GetNamedItemNS(namespaceURI, localName),
    FOwnerDocument);
end;

function TXMLPDOMNamedNodeMap.removeNamedItem(const name: DOMString): IDOMNode;
begin
  Result := MakeNode(NamedNodeMap.RemoveNamedItem(name), FOwnerDocument);
end;

function TXMLPDOMNamedNodeMap.removeNamedItemNS(
  const namespaceURI, localName: DOMString): IDOMNode;
begin
  Result := MakeNode(NamedNodeMap.RemoveNamedItemNS(namespaceURI, localName),
    FOwnerDocument);
end;

function TXMLPDOMNamedNodeMap.setNamedItem(const newItem: IDOMNode): IDOMNode;
begin
  Result := MakeNode(NamedNodeMap.SetNamedItem(GetNode(newItem)),
    FOwnerDocument);
end;

function TXMLPDOMNamedNodeMap.setNamedItemNS(const arg: IDOMNode): IDOMNode;
begin
  Result := MakeNode(NamedNodeMap.SetNamedItem(GetNode(arg)), FOwnerDocument);
end;

{ TXMLPDOMCharacterData -------------------------------------------------------}

function TXMLPDOMCharacterData.GetCharacterData: TXpCharacterData;
begin
  Result := Node as TXpCharacterData;
end;

procedure TXMLPDOMCharacterData.appendData(const data: DOMString);
begin
  CharacterData.AppendData(data);
end;

procedure TXMLPDOMCharacterData.deleteData(offset, count: Integer);
begin
  CharacterData.DeleteData(offset, count);
end;

function TXMLPDOMCharacterData.get_data: DOMString;
begin
  Result := CharacterData.Data;
end;

function TXMLPDOMCharacterData.get_length: Integer;
begin
  Result := CharacterData.Length;
end;

procedure TXMLPDOMCharacterData.insertData(offset: Integer;
  const data: DOMString);
begin
  CharacterData.InsertData(offset, data);
end;

procedure TXMLPDOMCharacterData.replaceData(offset, count: Integer;
  const data: DOMString);
begin
  CharacterData.ReplaceData(offset, count, data);
end;

procedure TXMLPDOMCharacterData.set_data(const data: DOMString);
begin
  CharacterData.Data := data;
end;

function TXMLPDOMCharacterData.substringData(offset, count: Integer): DOMString;
begin
  Result := CharacterData.SubStringData(offset, count);
end;

{ TXMLPDOMAttr ----------------------------------------------------------------}

function TXMLPDOMAttr.GetAttribute: TXpAttribute;
begin
  Result := Node as TXpAttribute;
end;

function TXMLPDOMAttr.get_name: DOMString;
begin
  Result := Attribute.Name;
end;

function TXMLPDOMAttr.get_ownerElement: IDOMElement;
begin
  Result := MakeNode(Attribute.OwnerElement, FOwnerDocument) as IDOMElement;
end;

function TXMLPDOMAttr.get_specified: WordBool;
begin
  Result := Attribute.Specified;
end;

function TXMLPDOMAttr.get_value: DOMString;
begin
  Result := Attribute.Value;
end;

procedure TXMLPDOMAttr.set_value(const attributeValue: DOMString);
begin
  Attribute.Value := attributeValue;
end;

{ TXMLPDOMElement -------------------------------------------------------------}

function TXMLPDOMElement.GetElement: TXpElement;
begin
  Result := Node as TXpElement;
end;

function TXMLPDOMElement.get_tagName: DOMString;
begin
  Result := Element.TagName;
end;

function TXMLPDOMElement.getAttribute(const name: DOMString): DOMString;
begin
  Result := Element.GetAttribute(name);
end;

function TXMLPDOMElement.getAttributeNS(
  const namespaceURI, localName: DOMString): DOMString;
begin
  Result := Element.GetAttributeNS(namespaceURI, localName);
end;

function TXMLPDOMElement.getAttributeNode(const name: DOMString): IDOMAttr;
begin
  Result := MakeNode(Element.GetAttributeNode(name), FOwnerDocument) as IDOMAttr;
end;

function TXMLPDOMElement.getAttributeNodeNS(
  const namespaceURI, localName: DOMString): IDOMAttr;
begin
  Result := MakeNode(Element.Attributes.GetNamedItemNS(namespaceURI, localName),
    FOwnerDocument) as IDOMAttr;
end;

function TXMLPDOMElement.getElementsByTagName(const name: DOMString):
  IDOMNodeList;
begin
  Result := MakeNodeList(Element.GetElementsByTagName(name), FOwnerDocument);
end;

function TXMLPDOMElement.getElementsByTagNameNS(
  const namespaceURI, localName: DOMString): IDOMNodeList;
begin
  Result := MakeNodeList(Element.GetElementsByTagNameNS(namespaceURI, localName),
    FOwnerDocument);
end;

function TXMLPDOMElement.hasAttribute(const name: DOMString): WordBool;
begin
  Result := Element.HasAttribute(name);
end;

function TXMLPDOMElement.hasAttributeNS(
  const namespaceURI, localName: DOMString): WordBool;
begin
  Result := Element.GetAttributeNodeNS(namespaceURI, localName) <> nil;
end;

procedure TXMLPDOMElement.removeAttribute(const name: DOMString);
begin
  Element.RemoveAttribute(name);
end;

function TXMLPDOMElement.removeAttributeNode(const oldAttr: IDOMAttr): IDOMAttr;
begin
  Result := MakeNode(Element.RemoveAttributeNode(
    GetNode(oldAttr) as TXpAttribute), FOwnerDocument) as IDOMAttr;
end;

procedure TXMLPDOMElement.removeAttributeNS(
  const namespaceURI, localName: DOMString);
begin
  Element.RemoveAttributeNS(namespaceURI, localName);
end;

procedure TXMLPDOMElement.setAttribute(const name, value: DOMString);
begin
  Element.SetAttribute(name, value);
end;

function TXMLPDOMElement.setAttributeNode(const newAttr: IDOMAttr): IDOMAttr;
begin
  Result := MakeNode(Element.SetAttributeNode(
    GetNode(newAttr) as TXpAttribute), FOwnerDocument) as IDOMAttr;
end;

function TXMLPDOMElement.setAttributeNodeNS(const newAttr: IDOMAttr): IDOMAttr;
begin
  Result := MakeNode(Element.SetAttributeNodeNS(
    GetNode(newAttr) as TXpAttribute), FOwnerDocument) as IDOMAttr;
end;

procedure TXMLPDOMElement.setAttributeNS(
  const namespaceURI, qualifiedName, value: DOMString);
begin
  Element.SetAttributeNS(namespaceURI, qualifiedName, value);
end;

procedure TXMLPDOMElement.normalize;
begin
  Element.Normalize;
end;

{ TXMLPDOMText ----------------------------------------------------------------}

function TXMLPDOMText.splitText(offset: Integer): IDOMText;
begin
  Result := MakeNode((Node as TXpText).SplitText(offset), FOwnerDocument)
    as IDOMText;
end;

{ TXMLPDOMDocumentTypeChildren ------------------------------------------------}

type
  TXMLPDOMDocumentTypeChildren = class(TInterfacedObject, IDOMNodeList)
  private
    FDocumentType: TXMLPDOMDocumentType;
  protected
    { IDOMNodeList }
    function get_item(index: Integer): IDOMNode; safecall;
    function get_length: Integer; safecall;
  public
    constructor Create(const DocumentType: TXMLPDOMDocumentType);
  end;

constructor TXMLPDOMDocumentTypeChildren.Create(
  const DocumentType: TXMLPDOMDocumentType);
begin
  inherited Create;
  FDocumentType := DocumentType;
end;

function TXMLPDOMDocumentTypeChildren.get_item(index: Integer): IDOMNode;
var
  Index1, Index2: Integer;
begin
  Index1 := 0;
  for Index2 := 0 to FDocumentType.DocumentType.ChildNodes.Length - 1 do
    if FDocumentType.DocumentType.ChildNodes.Item(Index2).NodeType in
        [ELEMENT_NODE..NOTATION_NODE, ENTITY_DECL_NODE, NOTATION_DECL_NODE] then
    begin
      if Index1 = index then
      begin
        Result := MakeNode(FDocumentType.DocumentType.ChildNodes.Item(Index2),
          FDocumentType.FOwnerDocument);
        Exit;
      end;
      Inc(Index1);
    end;
  Result := MakeNode(FDocumentType.DocumentType.ExternalDTD,
    FDocumentType.FOwnerDocument);
end;

function TXMLPDOMDocumentTypeChildren.get_length: Integer;
var
  Index: Integer;
begin
  Result := 0;
  for Index := 0 to FDocumentType.DocumentType.ChildNodes.Length - 1 do
    if FDocumentType.DocumentType.ChildNodes.Item(Index).NodeType in
        [ELEMENT_NODE..NOTATION_NODE, ENTITY_DECL_NODE, NOTATION_DECL_NODE] then
      Inc(Result);
  if Assigned(FDocumentType.DocumentType.ExternalDTD) then
    Inc(Result);
end;

{ TXMLPDOMNamedChildren -------------------------------------------------------}

type
  TXMLPDOMNamedChildren = class(TInterfacedObject, IDOMNamedNodeMap)
  private
    FDocumentType: TXMLPDOMDocumentType;
    FNodeType: Integer;
  protected
    { IDOMNamedNodeMap }
    function get_item(index: Integer): IDOMNode; safecall;
    function get_length: Integer;
    function getNamedItem(const name: DOMString): IDOMNode; safecall;
    function setNamedItem(const arg: IDOMNode): IDOMNode; safecall;
    function removeNamedItem(const name: DOMString): IDOMNode; safecall;
    function getNamedItemNS(const namespaceURI, localName: DOMString):
      IDOMNode; safecall;    { DOM Level 2 }
    function setNamedItemNS(const arg: IDOMNode): IDOMNode; safecall;
                             { DOM Level 2 }
    function removeNamedItemNS(const namespaceURI, localName: DOMString):
      IDOMNode; safecall;    { DOM Level 2 }
  public
    constructor Create(const DocumentType: TXMLPDOMDocumentType;
      NodeType: Integer);
  end;

constructor TXMLPDOMNamedChildren.Create(
  const DocumentType: TXMLPDOMDocumentType; NodeType: Integer);
begin
  inherited Create;
  FDocumentType := DocumentType;
  FNodeType     := NodeType;
end;

function TXMLPDOMNamedChildren.get_item(index: Integer): IDOMNode;
var
  Index1, Index2: Integer;
begin
  Index1 := 0;
  for Index2 := 0 to FDocumentType.DocumentType.ChildNodes.Length - 1 do
    if FDocumentType.DocumentType.ChildNodes.Item(Index2).NodeType = FNodeType then
    begin
      if Index1 = index then
      begin
        Result := MakeNode(FDocumentType.DocumentType.ChildNodes.Item(Index2),
          FDocumentType.FOwnerDocument);
        Exit;
      end;
      Inc(Index1);
    end;
  Result := nil;
end;

function TXMLPDOMNamedChildren.get_length: Integer;
var
  Index: Integer;
begin
  Result := 0;
  for Index := 0 to FDocumentType.DocumentType.ChildNodes.Length - 1 do
    if FDocumentType.DocumentType.ChildNodes.Item(Index).NodeType = FNodeType then
      Inc(Result);
end;

function TXMLPDOMNamedChildren.getNamedItem(const name: DOMString): IDOMNode;
var
  Index: Integer;
begin
  Result := nil;
  for Index := 0 to FDocumentType.DocumentType.ChildNodes.Length - 1 do
    if (FDocumentType.DocumentType.ChildNodes.Item(Index).NodeType = FNodeType) and
        (FDocumentType.DocumentType.ChildNodes.Item(Index).NodeName = name) then
    begin
      Result := MakeNode(FDocumentType.DocumentType.ChildNodes.Item(Index),
        FDocumentType.FOwnerDocument);
      Exit;
    end;
end;

function TXMLPDOMNamedChildren.getNamedItemNS(const namespaceURI,
  localName: DOMString): IDOMNode;
var
  Index: Integer;
begin
  Result := nil;
  for Index := 0 to FDocumentType.DocumentType.ChildNodes.Length - 1 do
    if (FDocumentType.DocumentType.ChildNodes.Item(Index).NodeType = FNodeType) and
        (FDocumentType.DocumentType.ChildNodes.Item(Index).NamespaceURI = namespaceURI) and
        (FDocumentType.DocumentType.ChildNodes.Item(Index).NodeName = localName) then
    begin
      Result := MakeNode(FDocumentType.DocumentType.ChildNodes.Item(Index),
        FDocumentType.FOwnerDocument);
      Exit;
    end;
end;

function TXMLPDOMNamedChildren.removeNamedItem(const name: DOMString): IDOMNode;
begin
  DOMVendorNotSupported('removeNamedItem', SXMLPartner); { Do not localize }
end;

function TXMLPDOMNamedChildren.removeNamedItemNS(const namespaceURI,
  localName: DOMString): IDOMNode;
begin
  DOMVendorNotSupported('removeNamedItemNS', SXMLPartner); { Do not localize }
end;

function TXMLPDOMNamedChildren.setNamedItem(const arg: IDOMNode): IDOMNode;
begin
  DOMVendorNotSupported('setNamedItem', SXMLPartner); { Do not localize }
end;

function TXMLPDOMNamedChildren.setNamedItemNS(const arg: IDOMNode): IDOMNode;
begin
  DOMVendorNotSupported('setNamedItemNS', SXMLPartner); { Do not localize }
end;

{ TXMLPDOMDocumentType --------------------------------------------------------}

constructor TXMLPDOMDocumentType.Create(const Node: TXpNode;
  const Document: TXMLPDOMDocument);
begin
  inherited Create(Node, Document);
  FChildren  := TXMLPDOMDocumentTypeChildren.Create(Self);
  FEntities  := TXMLPDOMNamedChildren.Create(Self, ENTITY_DECL_NODE);
  FNotations := TXMLPDOMNamedChildren.Create(Self, NOTATION_DECL_NODE);
end;

function TXMLPDOMDocumentType.GetDocumentType: TXpDocumentType;
begin
  Result := Node as TXpDocumentType;
end;

function TXMLPDOMDocumentType.get_childNodes: IDOMNodeList;
begin
  Result := FChildren;
end;

function TXMLPDOMDocumentType.get_entities: IDOMNamedNodeMap;
begin
  Result := FEntities;
end;

function TXMLPDOMDocumentType.get_internalSubset: DOMString;
begin
  Result := DocumentType.XmlDocument;
end;

function TXMLPDOMDocumentType.get_name: DOMString;
begin
  Result := DocumentType.Name;
end;

function TXMLPDOMDocumentType.get_notations: IDOMNamedNodeMap;
begin
  Result := FNotations;
end;

function TXMLPDOMDocumentType.get_publicId: DOMString;
begin
  Result := DocumentType.PublicID;
end;

function TXMLPDOMDocumentType.get_systemId: DOMString;
begin
  Result := DocumentType.SystemID;
end;

function TXMLPDOMDocumentType.hasChildNodes: WordBool;
begin
  Result := (get_childNodes.Length > 0);
end;

{ TXMLPDOMNotation ------------------------------------------------------------}

function TXMLPDOMNotation.GetNotation: TXpDTDNotation;
begin
  Result := Node as TXpDTDNotation;
end;

function TXMLPDOMNotation.get_nodeType: DOMNodeType;
begin
  Result := NOTATION_NODE;
end;

function TXMLPDOMNotation.get_publicId: DOMString;
begin
  Result := Notation.PublicId;
end;

function TXMLPDOMNotation.get_systemId: DOMString;
begin
  Result := Notation.SystemId;
end;

{ TXMLPDOMEntity --------------------------------------------------------------}

function TXMLPDOMEntity.GetEntity: TXpDTDEntity;
begin
  Result := Node as TXpDTDEntity;
end;

function TXMLPDOMEntity.get_nodeType: DOMNodeType;
begin
  Result := ENTITY_NODE;
end;

function TXMLPDOMEntity.get_notationName: DOMString;
begin
  Result := Entity.NotationName;
end;

function TXMLPDOMEntity.get_publicId: DOMString;
begin
  Result := Entity.PublicId;
end;

function TXMLPDOMEntity.get_systemId: DOMString;
begin
  Result := Entity.SystemId;
end;

{ TXMLPDOMProcessingInstruction -----------------------------------------------}

function TXMLPDOMProcessingInstruction.GetProcessingInstruction:
  TXpProcessingInstruction;
begin
  Result := Node as TXpProcessingInstruction;
end;

function TXMLPDOMProcessingInstruction.get_data: DOMString;
begin
  Result := ProcessingInstruction.Data;
end;

function TXMLPDOMProcessingInstruction.get_target: DOMString;
begin
  Result := ProcessingInstruction.Target;
end;

procedure TXMLPDOMProcessingInstruction.set_data(const value: DOMString);
begin
  ProcessingInstruction.Data := value;
end;

{ TXMLPDOMDocument ------------------------------------------------------------}

constructor TXMLPDOMDocument.Create(const Node: TXpNode;
  const Document: TXMLPDOMDocument);
begin
  FXMLObjModel             := TXpObjModel.Create(nil);
  FXMLObjModel.RaiseErrors := False;
  if Assigned(Node) then
    inherited Create(Node, Self)
  else
    inherited Create(FXMLObjModel.Document, Self);
end;

destructor TXMLPDOMDocument.Destroy;
begin
  FXMLObjModel.Free;
  inherited Destroy;
end;

function TXMLPDOMDocument.GetDocument: TXpDocument;
begin
  Result := Node as TXpDocument;
end;

function TXMLPDOMDocument.createAttribute(const name: DOMString): IDOMAttr;
begin
  Result := IDOMAttr(MakeNode(Document.CreateAttribute(name), FOwnerDocument));
end;

function TXMLPDOMDocument.createAttributeNS(const namespaceURI,
  qualifiedName: DOMString): IDOMAttr;
begin
  Result := IDOMAttr(MakeNode(Document.CreateAttributeNS(
    namespaceURI, qualifiedName), FOwnerDocument));
end;

function TXMLPDOMDocument.createCDATASection(const data: DOMString):
  IDOMCDATASection;
begin
  Result := IDOMCDATASection(MakeNode(Document.CreateCDATASection(data),
    FOwnerDocument));
end;

function TXMLPDOMDocument.createComment(const data: DOMString): IDOMComment;
begin
  Result := IDOMComment(MakeNode(Document.CreateComment(data), FOwnerDocument));
end;

function TXMLPDOMDocument.createDocumentFragment: IDOMDocumentFragment;
begin
  Result := IDOMDocumentFragment(MakeNode(Document.CreateDocumentFragment,
    FOwnerDocument));
end;

function TXMLPDOMDocument.createElement(const tagName: DOMString): IDOMElement;
begin
  Result := IDOMElement(MakeNode(Document.CreateElement(tagName),
    FOwnerDocument));
end;

function TXMLPDOMDocument.createElementNS(
  const namespaceURI, qualifiedName: DOMString): IDOMElement;
begin
  Result := IDOMElement(MakeNode(Document.CreateElementNS(
    namespaceURI, qualifiedName), FOwnerDocument));
end;

function TXMLPDOMDocument.createEntityReference(const name: DOMString):
  IDOMEntityReference;
begin
  Result := IDOMEntityReference(MakeNode(Document.CreateEntityReference(name),
    FOwnerDocument));
end;

function TXMLPDOMDocument.createProcessingInstruction(
  const target, data: DOMString): IDOMProcessingInstruction;
begin
  Result := IDOMProcessingInstruction(MakeNode(
    Document.CreateProcessingInstruction(target, data), FOwnerDocument));
end;

function TXMLPDOMDocument.createTextNode(const data: DOMString): IDOMText;
begin
  Result := IDOMText(MakeNode(Document.CreateTextNode(data), FOwnerDocument));
end;

function TXMLPDOMDocument.get_doctype: IDOMDocumentType;
begin
  Result := IDOMDocumentType(MakeNode(Document.DocType, FOwnerDocument));
end;

function TXMLPDOMDocument.get_documentElement: IDOMElement;
begin
  Result := IDOMElement(MakeNode(Document.DocumentElement, FOwnerDocument));
end;

function TXMLPDOMDocument.get_domImplementation: IDOMImplementation;
begin
  Result := TXMLPDOMImplementation.Create(Document.DomImplementation);
end;

function TXMLPDOMDocument.get_nodeName: DOMString;
begin
  Result := '#document';
end;

function TXMLPDOMDocument.getElementById(const elementId: DOMString):
  IDOMElement;
begin
  Result := MakeNode(Document.SelectSingleNode('//*[@id="' + elementId + '"]'),
    FOwnerDocument) as IDOMElement;
end;

function TXMLPDOMDocument.getElementsByTagName(const tagName: DOMString):
  IDOMNodeList;
begin
  Result := MakeNodeList(Document.GetElementsByTagName(tagName), FOwnerDocument);
end;

function TXMLPDOMDocument.getElementsByTagNameNS(
  const namespaceURI, localName: DOMString): IDOMNodeList;
begin
  Result := MakeNodeList(Document.GetElementsByTagNameNS(
    namespaceURI, localName), FOwnerDocument);
end;

function TXMLPDOMDocument.importNode(importedNode: IDOMNode; deep: WordBool):
  IDOMNode;
begin
  Result := MakeNode(Document.ImportNode(GetNode(importedNode), deep),
    FOwnerDocument);
end;

procedure TXMLPDOMDocument.set_documentElement(const IDOMElement: IDOMElement);
begin
  if Assigned(Document.DocumentElement) then
    Document.RemoveChild(Document.DocumentElement);
  Document.AppendChild(GetNode(IDOMElement));
end;

{ IDOMParseOptions Interface }

function TXMLPDOMDocument.get_async: Boolean;
begin
  Result := False;
end;

function TXMLPDOMDocument.get_preserveWhiteSpace: Boolean;
begin
  Result := not FXMLObjModel.NormalizeData;
end;

function TXMLPDOMDocument.get_resolveExternals: Boolean;
begin
  Result := False;
end;

function TXMLPDOMDocument.get_validate: Boolean;
begin
  Result := FValidate;
end;

procedure TXMLPDOMDocument.set_async(Value: Boolean);
begin
  if Value <> get_async then
    DOMVendorNotSupported('set_async', SXMLPartner); { Do not localize }
end;

procedure TXMLPDOMDocument.set_preserveWhiteSpace(Value: Boolean);
begin
  FXMLObjModel.NormalizeData := not Value;
end;

procedure TXMLPDOMDocument.set_resolveExternals(Value: Boolean);
begin
  if Value <> get_resolveExternals then
    DOMVendorNotSupported('set_resolveExternals', SXMLPartner); { Do not localize }
end;

procedure TXMLPDOMDocument.set_validate(Value: Boolean);
begin
  FValidate := Value;
end;

{ IDOMPersist interface }

function TXMLPDOMDocument.asyncLoadState: Integer;
begin
  Result := 0;
end;

function TXMLPDOMDocument.get_xml: DOMString;
var
  Index: Integer;
begin
  Result := '';
  if FVersion <> '' then
    Result := Result + ' ' + sVersion + '="' + FVersion + '"';
  if FEncoding <> '' then
    Result := Result + ' ' + sEncoding + '="' + FEncoding + '"';
  if FStandalone <> '' then
    Result := Result + ' ' + sStandalone + '="' + FStandalone + '"';
  if Result <> '' then
    Result := '<?xml' + Result + '?>'#13;
  for Index := 0 to Document.ChildNodes.Length - 1 do
    Result := Result + Document.ChildNodes.Item(Index).XmlDocument + #13;
end;

function TXMLPDOMDocument.load(source: OleVariant): WordBool;
begin
  FGotProlog := False;
  Result     := FXMLObjModel.LoadDataSource(source);
  if not Result then
    Exit;
  FNode := FXMLObjModel.Document;
  if FValidate then
    Result := FXMLObjModel.ValidateDocument;
end;

function TXMLPDOMDocument.loadFromStream(const stream: TStream): WordBool;
begin
  FGotProlog := False;
  Result     := FXMLObjModel.LoadStream(Stream);
  if not Result then
    Exit;
  FNode := FXMLObjModel.Document;
  if FValidate then
    Result := FXMLObjModel.ValidateDocument;
end;

function TXMLPDOMDocument.loadxml(const Value: DOMString): WordBool;
var
  XML: DOMString;
begin
  FGotProlog := False;
  XML        := Value;
  Result     := FXMLObjModel.LoadMemory(XML[1], Length(XML));
  if not Result then
    Exit;
  FNode := FXMLObjModel.Document;
  if FValidate then
    Result := FXMLObjModel.ValidateDocument;
end;

procedure TXMLPDOMDocument.save(destination: OleVariant);
begin
  destination := Document.XmlDocument;
end;

procedure TXMLPDOMDocument.saveToStream(const stream: TStream);
var
  StrStream: TStringStream;
begin
  StrStream := TStringStream.Create(Document.XmlDocument);
  try
    Stream.CopyFrom(StrStream, 0);
  finally
    StrStream.Free;
  end;
end;

procedure TXMLPDOMDocument.set_OnAsyncLoad(const Sender: TObject;
  EventHandler: TAsyncEventHandler);
begin
  //
end;

{ IDOMParseError Interface }

function TXMLPDOMDocument.get_errorCode: Integer;
begin
  Result := 0;
end;

function TXMLPDOMDocument.get_filepos: Integer;
begin
  Result := -1;
end;

function FindEmbeddedValue(const Text, Name: WideString): WideString;
var
  Index: Integer;
  Value: WideString;
begin
  Result := '';
  if Text = '' then
    Exit;
  Value := Text;
  Index := Pos(Name, Value);
  if Index > 0 then
    Delete(Value, 1, Index + Length(Name) - 1);
  Index := Pos(' ', Value);
  if Index > 0 then
    Result := Copy(Value, 1, Index - 1);
end;

function TXMLPDOMDocument.get_line: Integer;
begin
  Result := -1;
  try
    Result := StrToInt(FindEmbeddedValue(get_reason, 'Line: '));
  except on EConvertError do
    // Ignore
  end;
end;

function TXMLPDOMDocument.get_linepos: Integer;
begin
  Result := -1;
  try
    Result := StrToInt(FindEmbeddedValue(get_reason, 'Col: '));
  except on EConvertError do
    // Ignore
  end;
end;

function TXMLPDOMDocument.get_reason: WideString;
begin
  Result := FXMLObjModel.Errors.Text;
end;

function TXMLPDOMDocument.get_srcText: WideString;
begin
  Result := '';
end;

function TXMLPDOMDocument.get_url: WideString;
begin
  Result := FindEmbeddedValue(get_reason, 'File: ');
end;

{ IDOMXMLProlog Interface }

procedure TXMLPDOMDocument.GetProlog;
var
  Data: string;
  Attrs: TStringList;

  procedure ExtractAttrs(const Data: DOMString; const Attrs: TStringList);
  var
    Index, Start, Len: Integer;
    Name: string;
    Quote: WideChar;
  begin
    Index := 1;
    Len   := Length(Data);
    repeat
      while (Index <= Len) and XPIsSpace(Ord(Data[Index])) do
        Inc(Index);
      Start := Index;
      while (Index <= Len) and XPIsNameChar(Ord(Data[Index])) do
        Inc(Index);
      Name := Copy(Data, Start, Index - Start);
      while (Index <= Len) and XPIsSpace(Ord(Data[Index])) do
        Inc(Index);
      if Data[Index] <> '=' then
        raise DOMException.Create('Expected "="');
      Inc(Index);
      while (Index <= Len) and XPIsSpace(Ord(Data[Index])) do
        Inc(Index);
      Quote := Data[Index];
      Inc(Index);
      Start := Index;
      while (Index <= Len) and (Data[Index] <> Quote) do
        Inc(Index);
      Attrs.Values[Name] := Copy(Data, Start, Index - Start);
      Inc(Index);
      while (Index <= Len) and XPIsSpace(Ord(Data[Index])) do
        Inc(Index);
    until Index > Length(Data);
  end;

begin
  if FGotProlog then
    Exit;
  FGotProlog := True;
  if (Document.FirstChild.NodeType <> PROCESSING_INSTRUCTION_NODE) or
      (TXpProcessingInstruction(Document.FirstChild).Target <> sXML) then
    Exit;
  Data  := TXpProcessingInstruction(Document.FirstChild).Data;
  Attrs := TStringList.Create;
  try
    ExtractAttrs(Data, Attrs);
    FVersion    := Attrs.Values[sVersion];
    FEncoding   := Attrs.Values[sEncoding];
    FStandalone := Attrs.Values[sStandalone];
  finally
    Attrs.Free;
  end;
end;

function TXMLPDOMDocument.get_Encoding: DOMString;
begin
  GetProlog;
  Result := FEncoding;
end;

function TXMLPDOMDocument.get_Standalone: DOMString;
begin
  GetProlog;
  Result := FStandalone;
end;

function TXMLPDOMDocument.get_Version: DOMString;
begin
  GetProlog;
  Result := FVersion;
end;

procedure TXMLPDOMDocument.set_Encoding(const Value: DOMString);
begin
  FEncoding := Value;
end;

procedure TXMLPDOMDocument.set_Standalone(const Value: DOMString);
begin
  FStandalone := Value;
end;

procedure TXMLPDOMDocument.set_Version(const Value: DOMString);
begin
  FVersion := Value;
end;

{ TXMLPDOMImplementationFactory -----------------------------------------------}

function TXMLPDOMImplementationFactory.DOMImplementation: IDOMImplementation;
begin
  Result := TXMLPDOMImplementation.Create(TXpDomImplementation.Create);
end;

function TXMLPDOMImplementationFactory.Description: String;
begin
  Result := SXMLPartner;
end;

initialization
  XPXML_DOM   := TXMLPDOMImplementationFactory.Create;
  RegisterDOMVendor(XPXML_DOM);
finalization
  UnRegisterDOMVendor(XPXML_DOM);
  XPXML_DOM.Free;
end.
